function PJ = hybridmodel(Uext,Y1,Y2,N,YN,fIDX);
% hybridmodel.m --- Mixed Modeling demo.
% 
% This function demonstrates a hybrid waveguide model consisting of N/2 K-nodes,
% N/2 W-nodes, and the KW converter, as shown in Figure 5. The excitation signal
% Uext is connected to the K-node N1 with appropriate filtering. Y1 and YN are
% termination admittances, and Y2 is the admittance of both the K-pipes and
% W-lines, The junction pressure PJ is returned. Optional <fIDX> flag shows an
% animation in figure(fIDX).
% 
% Excitation Examples: 
% Uext = [1 zeros(1,399)]; % Impulse
% Uext = [hamming(10)' zeros(1,390)]; % A smooth excitation signal
%
% Demo Examples:
% PH = hybridmodel(Uext,.10,1,50,.10); % Non-inverting terminations.
% PH = hybridmodel(Uext,5,1,50,10);    % Inverting terminations.
% PH = hybridmodel(Uext,.10,1,50,1);   % Admittance-matched termination (no
%                                   reflection from the right).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%% -*- Mode: Matlab -*- %%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Author          : Cumhur.Erkut@erase.hut.fi
%% Created On      : Wed Jun 30 12:33:22 2004
%% Last Modified By: Cumhur.Erkut@erase.hut.fi
%% Last Modified On: Thu Jul 01 12:10:02 2004
%% Update Count    : 208
%% Reference       : Digital Waveguides versus Finite Difference Structures: 
%%                   Equivalence and Mixed Modeling, Matti Karjalainen and
%%                   Cumhur Erkut. EURASIP J. of Applied Signal Processing,
%%                   Volume 2004, Number 7, pp. 978-989, 15 June 2004.
%%                   http://asp.hindawi.com/volume-2004/S1110865704401176.html
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ERROR CHECK
  error(nargchk(5,6,nargin));
  if nargin == 5
      fIDX = 0;
  end
%%%%%%%%%%%%%%%%%%%% SIMULATION PARAMETERS %%%%%%%%%%%%%%%%%%%%
 MAXSTEP = length(Uext);
 NK = fix(.5*N);
 NW = N - NK;
 Umax = max(Uext);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% -----------------  INITIALIZATION  --------------------------
 % Excitation filtering (Figure 3)
   Uext = filter([1 0 -1],1, Uext);

 % INITIALLY RELAXED STATES                                   
   P = zeros(3,NK);
   PJ = zeros(1,N);
 
 % DELAY LINES 
   P2 = 0; PN = 0; P1m = 0; PNp = 0;
   Pm = zeros(1,NW-1);
   Pp = zeros(1,NW-1);

 % KW-Converter memory
   kmem0 = 0;
   wmem0 = 0;
   wmem1 = 0;
% -------------------------------------------------------------

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% -----------------    RECURSION     --------------------------
  for n = 1:MAXSTEP
    
    % CALCULATE THE K-structure
    P(1,1) = 1/(Y1+Y2)*( 2* Y1 * P(3,1) + Uext(n) + 2* Y2 * P(2,2)) - P(3,1);
    P(1,2:NK-1) = P(2,1:NK-2) + P(2,3:NK) -  P(3,2:NK-1);
    P(1,NK) = P(2,NK-1) + kmem0 - P(3,NK);
    
    % CALCULATE THE W-nodes
    P2 = (P(2,NK) - wmem1 + Pp(1));
    PN = 1/(Y2+YN) * (2* Y2 * Pm(NW-1));
    
    % CALCULATE THE SCATTERED WAVES
    P1m  = P2 - Pp(1);
    PNp  = PN - Pm(NW-1);
    KWin =  Pp(1);

   % CALCULATE THE JUNCTION PRESSURES
    PJ = [P(1,:) P2 Pm(1:NW-2)+Pp(2:NW-1) PN];
    
    % UPDATE THE KW-Converter Memory
    kmem0 =  P2;
    wmem1 = wmem0;
    wmem0 = KWin;

    % UPDATE THE STATES
    P(3,:) = P(2,:);
    P(2,:) = P(1,:);
    
    % UPDATE DELAY LINES
    Pm = [P1m Pm(1:NW-2)];
    Pp = [Pp(2:NW-1) PNp];

 
    % % -------------    PLOT    -------------------
    if fIDX
      figure(fIDX);clf;
      set(gcf,'Renderer','OpenGL');

      % Junction Pressure
      h = plot(1:N,PJ);set(h,'LineWidth',2);
      grid on;
      set(gca,'XLim',[1 N],...
              'YLim',[-1.1*Umax 1.1*Umax],...
              'FontSize',14,...
              'FontName','TimesNewRoman');
      h = xlabel('Position k'); 
      ylabel('Pressure P_{J,k}');
      set(h,'VerticalAlignment','middle')
      title('Hybrid Model')
      drawnow;
      hold on; plot([NW NW],.75*[-Umax Umax],'r-.'); hold off;
      h=text(0.4*NK,.5*Umax,'K-model');
      h=text(NK+0.4*NW,.5*Umax,'W-model');
      h=text(0.85*NK,.85*Umax,'KW-Converter');
    end
  end
 
