function PJ = kmodel(Uext,Y1,Y2,N,YN,fIDX);
% kmodel.m --- Terminated K-delay line demo.
%
% This function demonstrates the terminated K-delay line consisting of N
% K-nodes, connected by K-pipes of equal admittance Y2, as shown in Figure 3
% (the third port of K-node N1 is omitted). The excitation signal Uext is
% connected to N1 with appropriate filtering. Y1 and YN are termination
% admittances. The junction pressure PJ is returned. Except in nodes N1 and NN,
% the admittance matched chain is calculated as in Figure 7. Optional <fIDX>
% flag shows an animation in figure(fIDX). 
%
% Excitation Examples: 
% Uext = [1 zeros(1,399)]; % Impulse
% Uext = [hamming(10)' zeros(1,390)]; % A smooth excitation signal
%
% Demo Examples:
% PK = kmodel(Uext,.10,1,50,.10); % Non-inverting terminations.
% PK = kmodel(Uext,5,1,50,10);    % Inverting terminations.
% PK = kmodel(Uext,.10,1,50,1);   % Admittance-matched termination (no
%                                   reflection from the right).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%% -*- Mode: Matlab -*- %%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Author          : Cumhur.Erkut@erase.hut.fi
%% Created On      : Wed Jun 30 12:33:22 2004
%% Last Modified By: Cumhur.Erkut@erase.hut.fi
%% Last Modified On: Wed Jun 30 13:13:05 2004
%% Update Count    : 43
%% Reference       : Digital Waveguides versus Finite Difference Structures: 
%%                   Equivalence and Mixed Modeling, Matti Karjalainen and
%%                   Cumhur Erkut. EURASIP J. of Applied Signal Processing,
%%                   Volume 2004, Number 7, pp. 978-989, 15 June 2004.
%%                   http://asp.hindawi.com/volume-2004/S1110865704401176.html
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ERROR CHECK
  error(nargchk(5,6,nargin));
  if nargin == 5
      fIDX = 0;
  end
%%%%%%%%%%%%%%%%%%%% SIMULATION PARAMETERS %%%%%%%%%%%%%%%%%%%%
 MAXSTEP = length(Uext);
 Umax = max(Uext);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% -----------------  INITIALIZATION  --------------------------
 % Excitation filtering (Figure 3)
   Uext = filter([1 0 -1],1, Uext);

 % INITIALLY RELAXED STATES                                   
   P = zeros(3,N);
% -------------------------------------------------------------

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% -----------------    RECURSION     --------------------------
  for n = 1:MAXSTEP 
    
    % CALCULATE THE FDTD
    P(1,1) = 1/(Y1+Y2)*( 2* Y1 * P(3,1) + Uext(n) + 2* Y2 * P(2,2)) - P(3,1);
    P(1,2:N-1) = P(2,1:N-2) + P(2,3:N) -  P(3,2:N-1);
    P(1,N) = 1/(YN+Y2)*( 2* YN * P(3,N) + 2* Y2 * P(2,N-1))  - P(3,N);
    PJ = P(1,:);

    % % -------------    PLOT    -------------------
    if fIDX
      figure(fIDX);clf;
      set(gcf,'Renderer','OpenGL');

      % Junction Pressure
      h = plot(1:N,PJ);set(h,'LineWidth',2);
      grid on;
      set(gca,'XLim',[1 N],...
              'YLim',[-1.1*Umax 1.1*Umax],...
              'FontSize',14,...
              'FontName','TimesNewRoman');
      h = xlabel('Position k'); 
          ylabel('Pressure P_{J,k}');
          set(h,'VerticalAlignment','middle')
          title('K-model')
      drawnow;
    end

    % UPDATE THE STATES
    P(3,:) = P(2,:);
    P(2,:) = P(1,:);
  end
 
