function PJ = wmodel(Uext,Y1,Y2,N,YN,fIDX);
% wmodel.m --- Terminated W-delay line demo.
%
% This function demonstrates the terminated W-delay line consisting of N
% W-nodes, connected by W-lines of equal admittance Y2, as shown in Figure 2
% (the third port of W-node N1 is omitted). The excitation signal Uext is
% connected to N1. Y1 and YN are termination admittances. The junction pressure
% PJ is returned. Except W-nodes N1 and NN, the junction pressures are obtained
% from delay lines. Optional <fIDX> flag shows an animation in figure(fIDX).
%
% Excitation Examples: 
% Uext = [1 zeros(1,399)]; % Impulse
% Uext = [hamming(10)' zeros(1,390)]; % A smooth excitation signal
%
% Demo Examples:
% PW = wmodel(Uext,.10,1,50,.10); % Non-inverting terminations.
% PW = wmodel(Uext,5,1,50,10);    % Inverting terminations.
% PW = wmodel(Uext,.10,1,50,1);   % Admittance-matched termination (no
%                                   reflection from the right).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%% -*- Mode: Matlab -*- %%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Author          : Cumhur.Erkut@erase.hut.fi
%% Created On      : Wed Jun 30 12:33:22 2004
%% Last Modified By: Cumhur.Erkut@erase.hut.fi
%% Last Modified On: Wed Jun 30 13:13:05 2004
%% Update Count    : 43
%% Reference       : Digital Waveguides versus Finite Difference Structures: 
%%                   Equivalence and Mixed Modeling, Matti Karjalainen and
%%                   Cumhur Erkut. EURASIP J. of Applied Signal Processing,
%%                   Volume 2004, Number 7, pp. 978-989, 15 June 2004.
%%                   http://asp.hindawi.com/volume-2004/S1110865704401176.html
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ERROR CHECK
  error(nargchk(5,6,nargin));
  if nargin == 5
      fIDX = 0;
  end
%%%%%%%%%%%%%%%%%%%% SIMULATION PARAMETERS %%%%%%%%%%%%%%%%%%%%
 MAXSTEP = length(Uext);
 Umax = max(Uext);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% -----------------  INITIALIZATION  --------------------------
 % INITIALLY RELAXED STATES                                   
   P1 = 0; PN = 0; P1m = 0; PNp = 0;
   Pm = zeros(1,N-1);
   Pp = zeros(1,N-1);
% -------------------------------------------------------------

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% -----------------    RECURSION     --------------------------
  for n = 1:MAXSTEP 
    
    % CALCULATE THE JUNCTION PRESSURES
    P1 = 1/(Y1+Y2)*(Uext(n) + 2* Y2 * Pp(1));
    PN = 1/(Y2+YN)*( 2* Y2 * Pm(N-1));
    PJ = [P1 Pm(1:N-2)+Pp(2:N-1) PN];

    % CALCULATE THE DELAY INPUTS
    P1m = P1 - Pp(1);
    PNp = PN - Pm(N-1);
    
    % % -------------    PLOT    -------------------
    if fIDX
      figure(fIDX);clf;
      set(gcf,'Renderer','OpenGL');

      % Junction Pressure
      h = plot(1:N,PJ);set(h,'LineWidth',2);
      grid on;
      set(gca,'XLim',[1 N],...
              'YLim',[-1.1*Umax 1.1*Umax],...
              'FontSize',14,...
              'FontName','TimesNewRoman');
      h = xlabel('Position k'); 
          ylabel('Pressure P_{J,k}');
          set(h,'VerticalAlignment','middle')
          title('W-model')
      drawnow;
    end

    % UPDATE DELAY LINES
    Pm = [P1m Pm(1:N-2)];
    Pp = [Pp(2:N-1) PNp];
    
  end
 
