function [y, A, W]=egld_ica(mixedsig, s1, v1, s2, v2, s3, v3, s4, ...
			    v4, s5, v5);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EGLD-ICA algorithm for Independent Component Analysis
% 
% Input:
%   mixedsig = n*m matrix of the observed mixture
%   s1       = string corresponding for a parameter name (optional)
%   v1       = value of the paramater given in s1        (optional)
%   s2       = string corresponding for a parameter name (optional)
%   v2       = value of the paramater given in s2        (optional)
%   s3       = string corresponding for a parameter name (optional)
%   v3       = value of the paramater given in s3        (optional)
%   s4       = string corresponding for a parameter name (optional)
%   v4       = value of the paramater given in s4        (optional)
%   s5       = string corresponding for a parameter name (optional)
%   v5       = value of the paramater given in s4        (optional)
%
% Output:
%   y        = n*m matrix of the separated signals
%   A        = n*n matrix (estimated mixing matrix)
%   W        = n*n matrix (estimated demixing matrix)
%   
% egld_ica(mixedsig) estimates the independent components from given
% multidimensional signals. Each row of the matrix mixedsig is an
% observed signal.  EGLD_ICA employs maximum likelihood approach where 
% the distributions of source signals are iteratively estimated using
% the Extended Generalized Lambda Distribution (EGLD), see 
%
% Eriksson, J., Karvanen, J., and Koivunen, V.:
% "Source Distribution Adaptive Maximum Likelihood Estimation of ICA Model", 
% Proceedings of Second International Workshop on
% Independent Component Analysis and Blind Signal Separation,
% Helsinki 2000, pp. 227--232
%
% The optimization is done by Hyvarinen's fixed-point algorithm,
% see http://www.cis.hut.fi/projects/ica/fastica/.
%
% Optional parameters:
%   'epsilon'           Convergence criterion
%   'maxNumIterations'  Maximum number of iterations
%   'type'              Chooses the model used besides the GLD.
%                       Possible values:
%                       'egld'    (use both GLD and GBD) 
%                       'gld'     (use GLD always)
%                       'gldtanh' (use tanh when GLD not applicable)
%   'borderBase'
%   'borderSlope'       The minimum kurtosis value for switching to 
%                       the GBD (tanh) area. Thus the algorithm
%                       uses GLD for values
%                       kurtosis>borderBase+borderSlope*skewness^2, 
%                       and GBD (tanh) for the other values. 
%                       The default slope value 1.75 comes from the
%                       relation between theoretical minimum kurtosis and
%                       the corresponding skewness. 
%                       
% Example: 
%  ES=egld_ica(mixedsig,'epsilon',0.00005,'maxNumIterations',50,'type','gld');
%
% Copyright (c) Helsinki University of Technology,
% Signal Processing Laboratory,
% Jan Eriksson, Juha Karvanen, and Visa Koivunen.
%
% For details see the files readme.txt
% and gpl.txt provided within the same package.
%
% Last modified: 21.9.2000
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The lookup table for estimation of the EGLD parameters is loaded.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if not(exist('GLD_moment_table')),
  egld_use;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Default values for parameters. This corresponds to standard
% EGLD-ICA algorithm such that GLD is used for kurtosis values
% >2.2+1.75*skewness^2 and GBD for other possible values.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
epsilon=0.0001;
maxNumIterations=200;
type='egld';
borderBase=2.2;
borderSlope=1.75;

if(rem(nargin-1,2)==1)
  error('Optional parameters should always go in pairs');
else
  for i=1:(nargin-1)/2,
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Extract the name and the value of parameter number i (0<=i<=4).
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    str_param = eval (['s' int2str(i)]);
    val_param = eval (['v' int2str(i)]);
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Set the value of the parameter number i.
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if strcmp (str_param, 'epsilon')
      epsilon = val_param;
    elseif strcmp (str_param, 'maxNumIterations')
      maxNumIterations = val_param;
    elseif strcmp (str_param, 'type')
      type= val_param;
    elseif strcmp (str_param, 'borderBase')
      borderBase  = val_param;   
    elseif strcmp (str_param, 'borderSlope')
      borderSlope = val_param;
    else
      disp(['Warning: Parameter "' str_param '" is not defined!']);
      disp(' ');
    end
  end 
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Call for the actual algorithm doing the dirty job.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[y, A, W]=fasticaegld(mixedsig,type,epsilon,...
		      maxNumIterations,borderBase,borderSlope);

% The end of the function %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
