function [phi,phi2]=gbd_score(x,beta);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GBD_SCORE -  Calculates the score function and its derivative
% for a distribution from the Generalized Beta Distribution.
%
% Input:
%   x    = the realization matrix (sample values)
%   beta = the distribution parameters vector [beta(1) beta(2) beta(3) beta(4)]
%
% Output:
%   phi   = the score function matrix corresponding to x  
%   phi2  = the derivative of the score function matrix corresponding to x  
%
% Copyright (c) Helsinki University of Technology,
% Signal Processing Laboratory,
% Jan Eriksson, Juha Karvanen, and Visa Koivunen.
%
% For details see the files readme.txt
% and gpl.txt provided within the same package.
%
% Last modified: 5.9.2000
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if nargin<2, disp('Usege: gbd_score(x,beta)');
             error('Not enough input arguments.');  
end

delta=0.01;
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The values of x are forced to interval [beta(1)+delta,beta(1)+beta(2)-delta]
% to ensure the numerical stability.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
x=max((beta(1)+delta),min(x,beta(1)+beta(2)-delta));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The functions are given by the equations (22) and (23) in
%   Eriksson, J., Karvanen, J., and Koivunen, V.:
%   "Source Distribution Adaptive Maximum Likelihood Estimation of ICA Model", 
%   Proceedings of Second International Workshop on
%   Independent Component Analysis and Blind Signal Separation,
%   Helsinki 2000, pp. 227--232
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phi =(beta(2)*beta(3)-x.*(beta(3)+beta(4))+beta(1)*...
                       (beta(3)+beta(4)))./((x-beta(1)).*(-x+beta(1)+beta(2)));
phi2=-(beta(3)+beta(4))./((x-beta(1)).*(-x+beta(1)+beta(2)))...
     +phi./(-x+beta(1)+beta(2))-phi./(x-beta(1));

% The end of the function %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

