function [bias,devi,pwr]=biasdev_c0(cues, spow, ic, c0, ffcues)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%     Compute bias and deviations of ITD or ILD
%           for a specific value of c0
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  [bias,devi,pwr]=biasdev_c0(cues, spow, ic, c0, ffcues)
%
%  Input parameters:
%    cues:      ITD or ILD as a function of time
%    spow:      signal power as a function of time
%    ic:        IC as a function of time
%    c0:        Cue selection threshold
%    ffcues:    vector with all free field cues
%
%  Output parameters:
%  (The output parameters are vectors, where the first
%   entries are the values for each source separately
%   and the last entry is the total combined value)
%    bias:      ITD or ILD bias
%    devi:      ITD or ILD deviation
%    pwr:       Power of selected cues
%
%  Authors:
%    C. Faller and J. Merimaa 12-2003
%   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% init

N          = max(size(cues));
M          = max(size(ffcues));

% make row vector

if size(cues,1) > 1
  cues = cues';
end

% compute distance of each cue to all of the M free
% free field cues

for i = 1:M,
  distances(i,1:N) = cues - ffcues(i);
end

% find source indices with smallest distances and the 
% corresponding errors

if M > 1
  [err idx] = min(abs(distances));
else
  err = distances;
  idx = ones(size(distances));
end

% compute bias, power, and deviation for each source

for i = 1:M,
  mask = (idx == i).*(ic > c0);
  
  m = sum(cues.*spow.*mask)./sum(spow.*mask+1e-20);
  bias(i) = ffcues(i) - m;
  pwr(i)  = sum(mask.*spow)/sum(spow);
  devi(i) = sqrt(sum((cues-m).^2.*spow.*mask)./sum(spow.*mask+1e-20));
end

% compute weighted average for all sources

bias(M+1) =  sum(abs(bias(1:M)).*pwr(1:M)) / sum(pwr(1:M)+1e-20);
pwr(M+1)  =  sum(pwr(1:M));
devi(M+1) =  sum(devi(1:M).*pwr(1:M)) / sum(pwr(1:M)+1e-20);
            
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
