%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%       Example for using cue selection toolbox
%
%
%  Authors:
%    C. Faller and J. Merimaa 11-2004
%   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear

% audio parameters

offset       = 0;           % 0=no offset, >0=offset applied to audio file
numsamples   = 2.4*44100;   % 0=full length, >0=number of samples
sfreq        = 44100;       % sampling frequency

% auditory periphery parameters

fcenter      = 500;         % center frequency of critical band pair [Hz]
SPL          = 60;          % average sound pressure level of stimulus [dB]
maxitd       = 1.;          % range of itds considered [ms]
maxild       = 7;           % range of ilds considered [dB]
tau_ild      = 10;          % time constant for ild estimation [ms]
tau_itd      = 10;          % time constant for itd estimation [ms]
tau_ic       = 10;          % time constant for ic estimation [ms]
statoffset   = 400;         % offset until statistics are considered [ms]

% cue selection parameters (computation of c0)

criterion    = 0;        % criterion for cue selection:
c0           = 0.98;     %   criterion=0: use c0 directly
p0           = 0.1;      %   criterion=1: compute c0 such that selected power = p0
itddev0      = 0.03;     %   criterion=2: compute c0 such that itd deviation = itddev0
ilddev0      = 0.2;      %   criterion=3: compute c0 such that ild deviation = ilddev0

% analysis/plotting parameters

rangeild     = 5;           % ild plotting range [dB]
rangeitd     = 1.;          % itd plotting range [ms]
bins         = 30;          % resolution for computation of bias and deviation as
                            % a function of c0
res          = 30;          % resolution for PDF computation

% audio input files
%   format: N+1 filenames for analysis of N sources:
%           => first N filenames: are one-source-free-field audio files
%           => last filename:     concurrent sources and/or reverberation

pathaudio  = 'audio/';
infiles = [
'pulses_leadlag5ms1   '
'pulses_leadlag5ms2   '
'pulses_leadlag5msall2'
];
Nfiles = size(infiles,1);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% compute free field cues for each source and cues
% for composite sound
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

tau = [tau_ild tau_itd tau_ic];
statoffsetN = statoffset*sfreq/1000;
for file_idx = 1:Nfiles,

  % read audio file
  
  audioin = [pathaudio deblank(infiles(file_idx,:)) '.wav'];
  disp(['Processing audio file: ' audioin]);
  Nmax = wavread(audioin,'size');
  Nmax = Nmax(1);
  if numsamples == 0
    N = Nmax;
  else
    N = numsamples;
  end
  if N+offset > Nmax
    N = Nmax - offset;
  end
  range = [1+offset N+offset];
  [s,sfreq,BITS]=wavread(audioin,range);
  if size(s,2) == 1
    disp('Please specify stereo file!');
    return;
  end
  in1 = s(:,1)';
  in2 = s(:,2)';
  clear s;
  
  % apply perceptual model to compute binaural cues (ITD, ILD, IC)
  
  [ild itd ic pow]=ild_itd_ic(sfreq,fcenter,in1,in2,maxitd,maxild,tau,statoffsetN,SPL);
  
  % file_idx = 1...Nfiles-1:
  %   One-source-free-field scenarios: compute free field cues and PDFs
          
  if file_idx < Nfiles        
      mean_ild_ff(file_idx) = sum(ild.*pow) / sum(pow);
      mean_itd_ff(file_idx) = sum(itd.*pow) / sum(pow);
  end    
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% compute cue selection threshold c0
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if criterion == 1
  [c0,pw0] = c0_pwr(ic,pow,p0);
  disp(['p0 = ' num2str(p0,'%1.3f') '; c0 = ' num2str(c0,'%1.3f') '; p0 = ' num2str(pw0,'%1.3f') ' (solution)']);    
end
if criterion == 2
  itddev0 = 0.03;
  [c0,pw0,dev0]=c0_dev(itd,ic,pow,itddev0,mean_itd_ff);
  disp(['dev0 = ' num2str(itddev0,'%1.3f') '; p0 = ' num2str(pw0,'%1.3f') '; c0 = ' num2str(c0,'%1.3f') '; dev0 = ' num2str(dev0,'%1.3f') ' (solution)']);    
end
if criterion == 3
  ilddev0 = 0.2;
  [c0,pw0,dev0]=c0_dev(ild,ic,pow,ilddev0,mean_itd_ff);
  disp(['dev0 = ' num2str(ilddev0,'%1.3f') '; p0 = ' num2str(pw0,'%1.3f') '; c0 = ' num2str(c0,'%1.3f') '; dev0 = ' num2str(dev0,'%1.3f') ' (solution)']);    
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% plot cue selection in time domain
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure

g = plot_timeselect(1,1,ild,itd,ic,c0,sfreq,mean_ild_ff,mean_itd_ff,pow,.01,7,1.1,.8);
axes(g(1));
title([int2str(fcenter) 'Hz'])
legend(['c_0 = ' num2str(c0)],4)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% plot PDFs without and with cue selection
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure

% compute PDF without cue selection (c0 = 0)

[ild_itd_pdf,ild_pdf,itd_pdf,ildaxis,itdaxis]=pdf_c0(ild,itd,ic,pow,maxild,maxitd,res,0);

g = plot_pdf([1 2],1,ild_itd_pdf,ild_pdf,itd_pdf,ildaxis,itdaxis,mean_ild_ff,mean_itd_ff,rangeild,rangeitd,.3);
axes(g(1));
title('WITHOUT CUE SELECTION');

% compute PDF for cue selection with c0

[wild_itd_pdf,wild_pdf,witd_pdf,ildaxis,itdaxis]=pdf_c0(ild,itd,ic,pow,maxild,maxitd,res,c0);
  
g=plot_pdf([1 2],2,wild_itd_pdf,wild_pdf,witd_pdf,ildaxis,itdaxis,mean_ild_ff,mean_itd_ff,rangeild,rangeitd,.3);
axes(g(1));
title('WITH CUE SELECTION');
axes(g(2));
ylabel('p(\tau|c_1_2>c_0)  ')
axes(g(3));
xlabel('p(\DeltaL|c_1_2>c_0)  ')


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% plot bias and deviation as a function of c0
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% the above code computed the free field cues for each source, i.e.
% considering lead and lag as sources. We remove here free field
% cues corresponding to the lag, since we are interested in
% bias and deviation only with respect to lead

mean_ild_ff = mean_ild_ff(1);
mean_itd_ff = mean_itd_ff(1);

% compute statistics     

ic0range = .8;
[bias1,dev1,pwr1,axs]=biasdev_c0_N(ild,pow,ic,ic0range,mean_ild_ff,bins);
[bias2,dev2,pwr2,axs]=biasdev_c0_N(itd,pow,ic,ic0range,mean_itd_ff,bins);

% plot results

figure
g=axes('Position',[0.12 0.6 .3 0.2]);
plot(axs,bias2,'k');
axis tight
set(g,'XTickLabel',[])
ylabel('b_{\tau} [ms]');
title('ITD bias and deviation')

g=axes('Position',[0.12 0.375 .3 0.2]);
plot(axs,dev2,'k');
axis tight
ylabel('\sigma_{\tau} [ms]');
set(g,'XTickLabel',[])

g=axes('Position',[0.12 0.25 .3 0.1]);
plot(axs,pwr1,'k');
axis tight
ylabel('p_0');
xlabel('c_0')

g=axes('Position',[0.6 0.6 .3 0.2]);
plot(axs,bias1,'k');
axis tight
set(g,'XTickLabel',[])
ylabel('b_{\Delta L} [dB]');
title('ILD bias and deviation')

g=axes('Position',[0.6 0.375 .3 0.2]);
plot(axs,dev1,'k');
axis tight
ylabel('\sigma_{\Delta L} [dB]');
set(g,'XTickLabel',[])

g=axes('Position',[0.6 0.25 .3 0.1]);
plot(axs,pwr1,'k');
axis tight
ylabel('p_0');
xlabel('c_0')


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
