function [ild,itd,ic,pow]=ild_itd_ic(sfreq,fcenter,in1,in2,maxitd,maxild,tau,ofst,spl)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%       Compute binaural cues (ILD, ITD, IC)
%          for one pair of critical bands
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  [ild,itd,ic,pow] = ild_itd_ic(sfreq,fcenter,in1,in2,maxitd,maxild,tau)
%
%  Input parameters:
%    sfreq:     sampling frequency [Hz]
%    fcenter:   center frequency of auditory filter [Hz]
%    in1:       left input signal
%    in2:       right input signal
%    maxild:    limit for ILD [dB]
%    maxitd:    limit for ITD [ms]
%    tau:       time constants [ms]
%               (one value = same time constant for ILD, ITD and IC)
%               (three values = different time constants for ILD, ITD and IC)
%    ofst:      offset applied when returning computed cues [samples] (useful 
%               for ignoring cues affected by startup process of model)
%    spl:       average sound pressure level of stimuli (initial zeros at signal
%               start are ignored for spl consideration)
%
%  Output parameters:
%    ild:       ILD as a function of time
%    itd:       ITD as a function of time
%    ic:        IC as a function of time
%    pow:       Signal power as a function of time
%
%
%  Authors:
%    C. Faller and J. Merimaa 07-2003
%   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

FAST = 1; % 0: only Matlab, 1: use C for acceleration


% parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

use_auditory_filter  = 1;      % 1: apply auditory filter and hair-cell model
plot_auditory_filter = 0;      % 1: plot auditory filter
addnoise             = 1;      % 1: add independent noise left and right


% init
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% % default spl
% 
% if spl == []
%   spl = 60;
% end
% 
% make row vectors

if size(in1,1) > 1
  in1 = in1';
end

if size(in2,1) > 1
  in2 = in2';
end

N = max(size(in1));

% init

if sum(size(tau)) == 2
  tau_ic       = tau;          % time constant for ic estimation [ms]
  tau_ild      = tau;          % time constant for ild estimation [ms]
  tau_itd      = tau;          % time constant for itd estimation [ms]
else
  tau_ild      = tau(1);       % time constant for ild estimation [ms]
  tau_itd      = tau(2);       % time constant for itd estimation [ms]
  tau_ic       = tau(3);       % time constant for ic estimation [ms]
end

maxitds     = round(maxitd*sfreq/1000);
alpha       = 1 / (tau_ic*sfreq/1000);
alpha2      = 1 / (tau_ild*sfreq/1000);
alpha3      = 1 / (tau_itd*sfreq/1000);

% apply auditory filter and hair-cell model

if use_auditory_filter == 1
    
  % generate and plot auditory filter
  
  EarQ   = 9.26449; %  Glasberg and Moore Parameters
  minBW  = 24.7;
  fcoefs = mmakeerbfilters(sfreq,fcenter,[], EarQ, minBW);  
  
  if plot_auditory_filter == 1
    
      % compute frequency responses of auditory filters
  
      y = merbfilterbank([1 zeros(1,1024)], fcoefs);
      resp = 20*log10(abs(fft(y')));
      freqScale = (0:1023)/1024*sfreq;
    
      figure
      subplot(2,1,1)
      semilogx(freqScale(1:511),resp(1:511));
      axis([100 sfreq/2 -60 0])
      xlabel('Frequency (Hz)'); ylabel('Filter Response (dB)');
      grid
  end
    
  % apply auditory filter
  
  in1 = merbfilterbank(in1, fcoefs);
  in2 = merbfilterbank(in2, fcoefs);
  
  % add noise
  
  if addnoise == 1
    
    % hearing threshold according to ISO 389 1975
    
    hthr = [
      % Hz      dB SPL (re: 2e-5 Pa)
        125     47.9
        250     28.3
        500     12.6
        1000    6.8
        1500    6.7
        2000    7.8
        3000    7.6
        4000    8.7
        6000    11.9
        8000    11.6
    ];
    
    [y i] = min(abs(hthr(:,1)-fcenter));
    thr = hthr(i,2);
          
    % add auditory filter to noise
      
    ns1 = merbfilterbank(randn(size(in1)), fcoefs);
    ns2 = merbfilterbank(randn(size(in2)), fcoefs);
    
    % ingore zero offset at beginning of signal
    
    sizein = max(size(in1));
    offset1 = sizein - max((abs(in1) > 0).*(sizein:-1:1)) + 1;
    offset2 = sizein - max((abs(in2) > 0).*(sizein:-1:1)) + 1;
    
    % compute power of both channels
    
    power1 = sum(in1(offset1:sizein).^2);
    power2 = sum(in2(offset2:sizein).^2);
    
    % take average as reference
    
    pwr    = (power1 + power2) / 2;
    offset = round((offset1 + offset2) / 2);
    
    % add noise
    
    gain = sqrt(pwr/sum(ns1(offset:sizein).^2)) .* 10^((thr-spl)/20);
    in1 = in1 + gain.*ns1;
    
    gain = sqrt(pwr/sum(ns2(offset:sizein).^2)) .* 10^((thr-spl)/20);
    in2 = in2 + gain.*ns2;
  end
  
  % apply neural transduction
  
  [in1, output_powervector, output_maxvector] = mmonauraltransduction(in1, 'envelope', sfreq, 1);
  [in2, output_powervector, output_maxvector] = mmonauraltransduction(in2, 'envelope', sfreq, 1);
  in1 = in1.*(in1 > 0);
  in2 = in2.*(in2 > 0);
    

  % run model
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
  if FAST == 0
    num     = 1e-40*ones(1,2*maxitds+1);
    den1    = 1e-40*ones(1,2*maxitds+1);
    den2    = 1e-40*ones(1,2*maxitds+1);
    num_    = 1e-40*ones(1,2*maxitds+1);
    den1_   = 1e-40*ones(1,2*maxitds+1);
    den2_   = 1e-40*ones(1,2*maxitds+1);
    lev1    = 1e-40*ones(1,2*maxitds+1);
    lev2    = 1e-40*ones(1,2*maxitds+1);
      
    ic     = zeros(1,N-2*maxitds);
    pow    = zeros(1,N-2*maxitds);
    ild    = zeros(1,N-2*maxitds);
    itd    = zeros(1,N-2*maxitds);
    for i = 1:(N-2*maxitds),
      
      if mod(i, sfreq/10) == 0
        disp(['  Processed ' num2str(i/sfreq) 's of audio']);
      end
    
      S1(1:(maxitds+1))             = in1(i+maxitds);
      S2(1:(maxitds+1))             = in2(i:(i+maxitds));
      S1((maxitds+1):(2*maxitds+1)) = in1((i+maxitds):-1:i);
      S2((maxitds+1):(2*maxitds+1)) = in2(i+maxitds);
      
      % estimate icc
      
      num  = alpha*S1.*S2 + (1-alpha)*num;
      den1 = alpha*S1.*S1 + (1-alpha)*den1;
      den2 = alpha*S2.*S2 + (1-alpha)*den2;
        
      ic_func          = num./sqrt(den1.*den2      + 1e-20   );
      [ic_max max_idx] = max(ic_func);
      ic(i)            = ic_max;
          
      % estimate itd
      
      num_  = alpha3*S1.*S2 + (1-alpha3)*num_;
      den1_ = alpha3*S1.*S1 + (1-alpha3)*den1_;
      den2_ = alpha3*S2.*S2 + (1-alpha3)*den2_;
        
      ic_func          = num./sqrt(den1.*den2+1e-20);
      [ic_max max_idx] = max(ic_func);
      
      itd(i) = (max_idx-maxitds-1)/sfreq*1000;
      
      % estimate ild and pow
      
      lev1 = alpha2*S1.^2 + (1-alpha2)*lev1;
      lev2 = alpha2*S2.^2 + (1-alpha2)*lev2;
      
      idx = max_idx;
      %idx = maxitds + 1;
      
      ild(i) = 10*log10((lev1(idx)+1e-40)/(lev2(idx)+1e-40));
      if abs(ild(i)) > maxild
        ild(i) = sign(ild(i))*maxild;
      end
    
      s1 = S1(idx);
      s2 = S2(idx);
      pow(i) = lev1(idx) + lev2(idx);
    end
  else
  
    N2 = N-2*maxitds;
  
    % write audio data to disk 
    
    fid = fopen('audio.raw', 'w');
    if fid == 0 | fid == -1
      disp('Could not create audio.raw file');
      return;
    end
    fwrite(fid,[in1 in2],'float');
    fclose(fid);
    
    % build command-line and exectute
    
    if isunix
      if strcmp(computer,'MAC') == 0
        cmd = ['./ilditdicLINUX86 ' int2str(sfreq) ' ' num2str(maxild) ' ' int2str(maxitds)];
      else
        cmd = ['./ilditdicMACOSX ' int2str(sfreq) ' ' num2str(maxild) ' ' int2str(maxitds)];
      end
    elseif ispc
      cmd = ['ilditdicWIN86 ' int2str(sfreq) ' ' num2str(maxild) ' ' int2str(maxitds)];
    else
      cmd = ['./ilditdic ' int2str(sfreq) ' ' num2str(maxild) ' ' int2str(maxitds)];
    end

    cmd = [cmd ' ' num2str(alpha) ' ' num2str(alpha2) ' ' num2str(alpha3) ' audio.raw cues.raw'];
    
    disp(cmd);
    eval(['! ' cmd]);
    
    % read computed cues from disk
            
    fid = fopen('cues.raw', 'r');
    if fid == 0 | fid == -1
      disp('Could not read cues.raw file');      
      if isunix
        eval(['! rm -f audio.raw cues.raw']);
      elseif ispc
        eval(['! del audio.raw cues.raw']);
      else
        disp('Unknown platform: Did not delete temporary files.');
      end
      return;
    end
    data = fread(fid,'float');
    data = data';
    fclose(fid);
      
    % put data in correpsonding arrays
    
    if max(size(data)) ~= 4*N2
      disp('Data file not correct length!');
      if isunix
        eval(['! rm -f audio.raw cues.raw']);
      elseif ispc
        eval(['! del audio.raw cues.raw']);
      else
        disp('Unknown platform: Did not delete temporary files.');
      end
      return
    end
    
    k = 0;
    start = 1 + k*N2;
    stop  = (k+1)*N2;
    ild = data(start:stop);
    k = 1;
    start = 1 + k*N2;
    stop  = (k+1)*N2;
    itd = data(start:stop);
    k = 2;
    start = 1 + k*N2;
    stop  = (k+1)*N2;
    ic = data(start:stop);
    k = 3;
    start = 1 + k*N2;
    stop  = (k+1)*N2;
    pow = data(start:stop);
      
    % remove temporary files
      
    if isunix
      eval(['! rm -f audio.raw cues.raw']);
    elseif ispc
      eval(['! del audio.raw cues.raw']);
    else
      disp('Unknown platform: Did not delete temporary files.');
    end
  end
end

% apply offset

N = max(size(ild));
idx = (1+ofst):N;
ild = ild(idx);
itd = itd(idx);
ic  = ic(idx);
pow = pow(idx);
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
