function [ild_itd_pdf,ild_pdf,itd_pdf,axis_ild,axis_itd]=pdf_c0(ild,itd,ic,pow,maxild,maxitd,res,c0)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%     Compute ILD-ITD probability density functions:
%      p(ILD,ITD|IC>c0), p(ILD|IC>c0), p(ITD|IC>c0)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  [ild_itd_pdf,ild_pdf,itd_pdf,axis_ild,axis_itd] = pdf_c0(ild,itd,ic,pow,maxild,maxitd,res,c0)
%
%  Input parameters:
%    ild:       ILD as a function of time
%    itd:       ITD as a function of time
%    ic:        IC as a function of time
%    pow:       Signal power as a function of time
%    maxild:    limit for ILD [dB]
%    maxitd:    limit for ITD [ms]
%    res:       resolution of function computed
%    c0:        only cues with ic > c0 are considered for PDF
%
%  Output parameters:
%    ild_itd_pdf: p(ILD,ITD|IC>c0)
%    ild_pdf:     p(ILD|IC>c0)
%    itd_pdf:     p(ITD|IC>c0)
%    axis_ild:    ild values
%    axis_itd:    itd values
%
%
%  Authors:
%    C. Faller and J. Merimaa 12-2003
%   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

DITHER = 0; % 1: apply dithering for nicer graphs

f_ild    = maxild / res;
f_itd    = maxitd / res;
axis_ild = (-res:res).*f_ild;
axis_itd = (-res:res).*f_itd;

% remove values (by setting their power to zero) which
% are outside of plotting range

mask   = (abs(ild) <= maxild);
pow    = mask .* pow;
mask   = (abs(itd) <= maxitd);
pow    = mask .* pow;

% convert ild and itd to integers 1...(2*res+1)

N      = max(size(ild));
if DITHER == 1
  dither = rand(size(ild)) - .5;
else
  dither = 0;
end
ild    = round(ild ./ f_ild + dither) + res + 1;
if DITHER == 1
  dither = rand(size(itd)) - .5;
else
  dither = 0;
end
itd    = round(itd ./ f_itd + dither) + res + 1;
pw0    = .2.*mean(pow);

M = 2*res+1;
ild_itd_pdf = zeros(M,M);

% "mask" is a vector with zeros for unselected (non-considered)
% cues and ones for selected cues

mask  = (ic > c0).*(pow > pw0);
vals  = mask.*(ild + (M+1)*itd);
bins  = 0:((M+1)^2-1);
pdf   = hist(vals,bins);

for i = 1:M,
   start = 2+i*(M+1);
   stop  = start+M-1;
   ild_itd_pdf(1:M,i) = pdf(start:stop)';
end

mask    = (ic > c0).*(pow > pw0);
vals    = mask.*ild;
bins    = 0:M;
ild_pdf = hist(vals,bins);
ild_pdf = ild_pdf(2:(M+1));

mask    = (ic > c0).*(pow > pw0);
vals    = mask.*itd;
bins    = 0:M;
itd_pdf = hist(vals,bins);
itd_pdf = itd_pdf(2:(M+1));

% normalize PDFs

ild_pdf      = ild_pdf/max(ild_pdf+1e-40);
itd_pdf      = itd_pdf/max(itd_pdf+1e-40);
ild_itd_pdf  = ild_itd_pdf/max(max(ild_itd_pdf+1e-40));
            
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
