function [g]=plot_timeselect(psize,pos,ild,itd,ic,c0,sfreq,ild_ff,itd_ff,pwr,pthr,ildrange,itdrange,minic)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%     Plot ITD, ILD, and IC as a function of time
%        and mark selected ITD and ILD cues
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% [g] = plot_timeselect(psize,pos,ild,itd,ic,pwr,ild_ff,itd_ff,ildrange,itdrange,minic,sfreq,c0,pthr)
%
%  Input parameters:
%    psize:     plot size (1, 2, or 3 columns fit in window)
%    pos:       plot positions (1: left, >1: right)
%    ild:       ILD as a function of time
%    itd:       ITD as a function of time
%    ic:        IC as a function of time
%    c0:        cue selection threshold
%    sfreq:     sampling frequency
%    ild_ff:    free field ILDs (optional)
%    itd_ff:    free field ITDs (optional)
%    pwr:       power as a function of time (optional for selection)
%    pthr:      relative power threshold (optional for selection)
%    ildrange:  plot range for ILD [dB] (optional,
%               default: ildrange = max(abs(ild)))
%    itdrange:  plot range for ITD [ms] (optional,
%               default: itdrange = max(abs(itd)))
%    minic:     plot range for IC = [minic, 1.05] (optional,
%               default: minic = min(ic)-0.05)
%
%    Note: ildrange, itdrange, and minic should always be specified
%          and equal in plots including several columns. Otherwise the
%          y-axis scale labels might not correspond to the rightmost
%          columns.
%          
%
%  Output parameters:
%    g:         vector with handles to axis of
%               each plot
%
%  Authors:
%    C. Faller and J. Merimaa 12-2003
%   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if nargin < 8
  ild_ff = [];
end
if nargin < 9
  itd_ff = [];
end
if nargin < 10
  pwr = [];
end
if nargin < 11 | isempty(pthr)
  pthr = 0;
end
if nargin < 12 | isempty(ildrange)
  ildrange = max(abs(ild));
end
if nargin < 13 | isempty(itdrange)
  itdrange = max(abs(itd));
end
if nargin < 14 | isempty(minic)
  minic = max(0, min(ic)-0.05);
end

if psize == 1
  box1 = [0.13 0.68-0*.28 .8 0.25];
  box2 = [0.13 0.68-1*.28 .8 0.25];
  box3 = [0.13 0.68-2*.28 .8 0.25];
elseif psize == 2
  box1 = [0.15+(pos-1)*.42 0.75-0*.22 .4 0.19];
  box2 = [0.15+(pos-1)*.42 0.75-1*.22 .4 0.19];
  box3 = [0.15+(pos-1)*.42 0.75-2*.22 .4 0.19];
elseif psize == 3
  box1 = [0.08+(pos-1)*.27 0.75-0*.22 .25 0.19];
  box2 = [0.08+(pos-1)*.27 0.75-1*.22 .25 0.19];
  box3 = [0.08+(pos-1)*.27 0.75-2*.22 .25 0.19];
else
  error(['psize = ' int2str(psize) ' not supported']);
end

plotkind  = 'k:';
plotwidth = 1.0;
idx       = 1:(max(size(ild)));
timesc    = 1/sfreq;
numsrc    = max(size(ild_ff));

% ic plot
g(1)=axes('Position',box1);
plot(timesc.*(idx-1), ones(size(idx)).*c0,'k--','linewidth',.5);
hold on
plot(timesc.*(idx-1), ic(idx),plotkind,'linewidth',plotwidth);
hold off
axis([0 max(idx-1)*timesc minic 1.05])
set(g,'XTickLabel',[])
if pos > 1
  set(g,'YTickLabel',[])
else
  ylabel('c_1_2')
end

% itd plot
g(2)=axes('Position',box2);
plot(timesc.*(idx-1), itd(idx),plotkind,'linewidth',plotwidth);
hold on
% free-field cues
if ~isempty(itd_ff)
  for i = 1:numsrc,
    plot(timesc.*(idx-1), ones(size(idx)).*itd_ff(i),'k--','linewidth',.5);
  end
end
% highlight selected cues
done = 0;
mask = (ic(idx) > c0);
if pthr > 0 & ~isempty(pwr)
  mask = mask & (pwr(idx) > pthr*mean(pwr(idx)));
end
Nidx = max(size(idx));
i = 1;
while ~done
  while mask(i) == 0 & ~done,
    i = i + 1;
    if i >= Nidx
      done = 1;
    end
  end
  start = i;
  while mask(i) == 1 & ~done,
    i = i + 1;
    if i >= Nidx
      done = 1;
    end
  end
  stop = i - 1;
  plot(timesc.*(idx(start:stop)-1), itd(idx(start:stop)),'k','linewidth',2.5);
end
hold off
axis([0 max(idx-1)*timesc -itdrange itdrange])
if pos > 1
  set(g,'YTickLabel',[])
else
  ylabel('\tau [ms]')
end
set(g,'XTickLabel',[])

% itd plot
g(3)=axes('Position',box3);
plot(timesc.*(idx-1), ild(idx),plotkind,'linewidth',plotwidth);
hold on
if ~isempty(ild_ff)
  for i = 1:numsrc,
    plot(timesc.*(idx-1), ones(size(idx)).*ild_ff(i),'k--','linewidth',.5);
  end
end
% highlight the selected cues
done = 0;
i = 1;
while ~done
  while mask(i) == 0 & ~done,
    i = i + 1;
    if i >= Nidx
        done = 1;
    end
  end
  start = i;
  while mask(i) == 1 & ~done,
    i = i + 1;
    if i >= Nidx
        done = 1;
    end
  end
  stop = i - 1;
  plot(timesc.*(idx(start:stop)-1), ild(idx(start:stop)),'k','linewidth',2.5);
end
hold off
axis([0 max(idx-1)*timesc -ildrange ildrange])
if pos > 1
  set(g,'YTickLabel',[])
else
  ylabel('\Delta L [dB]')
end
xlabel('TIME [s]')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
