function [E,Q]=braun1(N,I,X) 

% An iterative method to optimize the allpass (AP) structure
% defining a real Kautz filter utilizing a FIR-to-IIR conversion
% method proposed by Brandenstein and Unbehauen, 1998.
%
% [E,Q]=braun1(N,I,X): N is the desired AP/Kautz filter order, 
% I iterations (e.g. 30), X is the target response. Braun1 
% produces iteratively (i=1:I) AP denominator polynomial 
% coefficients -> Q(i,:) and corresponding normalized  RMS 
% errors E(i,1) (this is really the true LS approximation error).
%
% Comments: if L>>N, the matrix A may become singular at some 
% iterations: warnings occur and E peaks to inf or NaN but we should
% still get useful results! You might however try to truncate X. 
% min(E) is not necessarily the  convergence value. Observe E and 
% use [m,k]=min(E) to get poles=roots(Q(k,:)). Increase I if needed.

L=length(X); 			% Implicitly L>N! In B&U FIR order is L+1 
A=zeros(L,N);			% but AP order is L? 
b=zeros(L,1);
q=[1 zeros(1,N)];			% The initial AP filter is 
								% A(z)=z^-N*q(1/z)/q(z)=z^-N,
X=X(L:-1:1);         	% time inverse of X               
                        
for i=1:I
   x=filter(1,q,X);								% all-pole-filtering with 1/q
   b=-[zeros(N,1);x(1:L-N)];					% Forming the matrix equation	
   for k=1:N										% Aq=u+b to minimize u=Aq-b
      A(:,k)=[zeros(k-1,1);x(1:L-k+1)];	% with Aq=b, e.i. we don't 
   end												% actually use u!	
   qi=A\b;							% AP denominator polynomial coefficients
   q=[1 qi(N:-1:1)'];			% are the mirror polynomial coefficients
   iq=q(N+1:-1:1);				% numerator z^-N*q(1/z) coefficients
   u=filter(iq,q,X);				% AP filtering
   Q(i,:)=q;							 % q's in rows of Q
   E(i,1)=sqrt(u'*u/(X'*X));			% normalized RMSE - in B&U
end											% u=u(1:N-1) ?!

% Revised November 7, 2000, Tuomas Paatero 
