function [c,EI,h]=kautzapprx2(a,M,x)

% [c,EI,h]=kautzapprx2(a,M,x): Approximates (in the LS sense) x with a real 
% Kautz filter constructed from M identical AP block defined by pole vector a. 
%
% Note that a is replaced by a=a(find(imag(a)>=0)) to avoid duplication 
% produced by complex conjugate poles, so check the resulting filter order. 
%
% Unfortunately we have to calculate the Kautz responses internally, twice, 
% to keep track of the approximation error with respect to the blocks.
%
% c are the Fourier coefficients produced by feeding the time-inverted
% x to the Kautz filter and reading tap-outputs at time N. These are 
% the LS optimal coefficients for approximating x with the Kautz model
% h=GM*c, where GM are the corresponding tap output impulse responses.
%
% E(i), i=1:M, is the normalized RMS error as a function of blocks used
% in the approximation, which enable a trivial model reduction scheme...
%
% Note that this is the "true" LS estimate in [1,infty] (not in [1,N]). 

N=length(x);
a=a(:);						% Poles with imag(a)<0 are disregarded to avoid
a=a(find(imag(a)>=0));	% dublication for conjugate pole pairs in a.
ar=a(find(imag(a)==0)); % real poles
ak=a(find(imag(a)>0)); 	% complex poles
m=length(ar);					
n=length(ak);
L=m+2*n;						% order of the AP block
q=-2*real(ak);				% calculation of the 2nd order
r=abs(ak).^2;				% polynomial terms

A=x(N:-1:1); 				% Input to the Kautz filter is the time-inverse
B=[1;zeros(N-1,1)];		% of x and an impulse, respectively
SM=[];
GM=[];
for j=1:M
   for i=1:m													% first we calculate
   S(:,i)=sqrt(1-ar(i)^2)*filter(1,[1 -ar(i)],A); 	% block's related to
   A=filter([-ar(i) 1],[1 -ar(i)],A);					% real poles
	end
% and then tap-output pairs corresponding to complex (conjugate) poles
	for i=1:n
		S(:,m+2*i-1)=sqrt((1-r(i))*(1+r(i)-q(i))/2)*filter([1 -1],[1 q(i) r(i)],A);
		S(:,m+2*i)=sqrt((1-r(i))*(1+r(i)+q(i))/2)*filter([1 1],[1 q(i) r(i)],A);   
   	A=filter([r(i) q(i) 1],[1 q(i) r(i)],A);
   end
   for i=1:m													% first we calculate
   	G(:,i)=sqrt(1-ar(i)^2)*filter(1,[1 -ar(i)],B); 	% block's related to
   	B=filter([-ar(i) 1],[1 -ar(i)],B);					% real poles
	end
% and then tap-output pairs corresponding to complex (conjugate) poles
	for i=1:n
		G(:,m+2*i-1)=sqrt((1-r(i))*(1+r(i)-q(i))/2)*filter([1 -1],[1 q(i) r(i)],B);
		G(:,m+2*i)=sqrt((1-r(i))*(1+r(i)+q(i))/2)*filter([1 1],[1 q(i) r(i)],B);   
   	B=filter([r(i) q(i) 1],[1 q(i) r(i)],B);
   end
   SM=[SM S];   			% construction of the non-impulse
   GM=[GM G];				% and impulse response marices.   
end
c=SM(N,:);					% calculation of Fourier/LS coefficients
c=c(:);
h=GM*c;						% approximation in [1,N]
X2=x'*x;						% energy of x
EM=X2;
for j=1:M											% approximation error as
   EI(j,1)=EM-sum(c((j-1)*L+1:j*L).^2);	% a function of blocks
   EM=EI(j);										% used in the approximation
end								
EI=sqrt(EI/X2);	% Normalized RMSE for i=1:M, e.g. EI(M)=(X2-c'*c)/X2

% Revised November 10, 2000, Tuomas Paatero
