 function kautzdemo3

load kautzdemo;

disp('    ')
disp('Welcome to a demonstration of the included basic tools') 
disp('    ')
disp('   * compkautz.m   - tap-output responses of a complex Kautz filter')
disp('   * mixkautz1.m   - responses of a real structure with real/c.c. poles')
disp('   * braun1.m      - one way to optimize the pole positions')
disp('   * kautzapprx1.m - composes and evaluates the Kautz approximation')
disp('   * kautzapprx2.m - the same for structures with identical blocks')
disp('  ')
disp('for Kautz filter design.') 
disp('  ')
disp('To get more information about the scripts use MATLAB help or inspect')
disp('directly the extensively commented m-files.')
disp(' ')
disp('.... press any key to continue ...')
pause
disp('  ')
disp('C=compkautz(a,N,x) produces tap-output responses to input x of a complex')
disp('Kautz filter defined by a pole vector a, |a|<1. The input is zero-padded')
disp('to length N, if N > length(x). The signal matrix C has dimension (N,p),')
%pause(1)
disp('where p=length(a) is (defined to be) the filter order. To proceed, we have')
disp('to choose a pole vector. For example a=r*ones(N,1), |r|<1, would give the')
%pause(1)
disp('Laguerre structure, real or complex depending on r. The figure presents')
disp('some possible pole distributions for the order 20: poles with a) linearly')
disp('and b) logarithmically spaced angles in [0,pi] and radius 0.99, c) the')
disp('latter with radius inversely proportional to the angle, and d) randomly')
disp('generated poles in imag(z)>0 and 0.9<|z|<1.')
%
%pause(2)
%
clf
subplot(2,2,1)
p(:,1)=.99*linang100(3:5:98);
zplane(p(:,1))
title('a) lin. spaced angles, r=0.99 ')
subplot(2,2,2)
p(:,2)=.99*logang100(4:5:99);
zplane(p(:,2))
title('b) log. spaced angles, r=0.99')
subplot(2,2,3)
p(:,3)=logaki100(4:5:99);
zplane(p(:,3))
title('c) with inv. prop. radii')
subplot(2,2,4)
p(:,4)=(.899+rand(20,1)/10).*exp(j*pi*rand(20,1));
zplane(p(:,4))
title('d) random: imag(z)>0, .9<|z|<1')
%
disp(' ')
disp('.... press any key to continue...')
pause
disp('  ')
i=menu('Choose a pole vector:','Linearly spaced angles with radius 0.99','Logarithmically spaced angles with radius 0.99','With radii inversely proportional to angles','Random poles in imag(z) > 0, 0.9 < |z| < 1','Your choice');
if i==5
   px=input('You may input any filter order and poles in |z|<1, but try proportion your \nchoice to the inherent dynamics proposed by the other choices\n\n>>');
   disp('  ')
   if any(abs(px)>=1)
      disp('!!!!!! Your pole vector will produce an unstable Kautz filter!!!!!!')
      disp('  ')
   end
   C=compkautz(px,1000,1);
else
   C=compkautz(p(:,i),1000,1);
end
%
disp('We produce the tap-output impulse responses with C=compkautz(a,1000,1).')
disp('The figure presents real parts of the signals in two time frames. The')
disp('figure is a mess but it demonstrates two things, the overall exponential')
disp('decay of the signals and the individual oscillation modes and damping')
disp('factors associated to each signal.')
%
subplot(2,1,1)
plot(real(C))
title('Real parts of 1000 sample impulse responses of the Kautz filter tap-outputs')
xlabel('Samples')
subplot(2,1,2)
plot(real(C(1:50,:)))
title('The first 50 samples of the same signals')
xlabel('Samples')
disp(' ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('Quite unsurprisingly, the corresponding magnitude responses (displayed')
disp('in the figure,) are closely related to the pole positions. Truncation')
disp('effects may occur if your own pole choice produced impulse responses')
disp('effectively exeeding 1000 samples.')
%
M=abs(fft(C));
subplot(2,1,1)
f1=linspace(0,1,500);
plot(f1,20*log10(M(1:500,:)))
title('Magnitude responses on a linear frequency scale')
xlabel('Normalized frequency')
ylabel('Magnitude / dB')
axis([0 1 -30 30])
subplot(2,1,2)
semilogx(f1,20*log10(M(1:500,:)))
title('Magnitude responses on a logarithmic frequency scale')
xlabel('Normalized frequency')
ylabel('Magnitude / dB')
axis([.01 1 -30 30])
%
disp(' ')
disp('.... press any key to continue ...')
pause
%
disp(' ')
disp('For a sufficient signal length N, the tap-output signals are orthonormal,')
disp('i.e. C''*C=I, where I is the (p times p) identity matrix and '' denotes')
disp('complex conjugate transpose. The marix product implements time-domain')
disp('inner products of the tap-output signals, and more generally, i.e. for an')
disp('non-impulse input, C''*C is (defined here to be) the correlation matrix of')
disp('the tap-output signals. The figure demonstrates the "orthonormality degree"')
disp('for some correlation windows. (Actually it is the absolute value of the')
disp('correlation terms that is illustrated - the correlation terms are complex')
disp('although the limit matrix I is real.)')
subplot(2,2,1)
pcolor(abs(C(1:25,:)'*C(1:25,:)))
title('signal length 25')
subplot(2,2,2)
pcolor(abs(C(1:50,:)'*C(1:50,:)))
title('signal length 50')
subplot(2,2,3)
pcolor(abs(C(1:150,:)'*C(1:150,:)))
title('signal length 150')
subplot(2,2,4)
pcolor(abs(C(1:500,:)'*C(1:500,:)))
title('signal length 500')
disp(' ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('Compkautz produces real tap-output signals only in the case of real poles.')
disp('However, from a sequence of real or complex conjugate pole pairs it is')
disp('always possible to construct real Kautz filter structures.') 
disp(' ')
disp('R=mixkautz1(a,N,x) utilizes one possible solution with an additional')
disp('treatment for real poles: 1st order blocks corresponding to real poles')
disp('are processed separately and complex poles with imag(a)>0 produce 2nd')
disp('order blocks with dual tap-outputs defined by a complex conjugate pole')
disp('pair (a,a''). Poles with imag(a)<0 are disregarded to avoid duplication') 
disp('produced by conjugate pole pairs in pole vector a.')
disp(' ')
disp('.... press any key to continue ...')
pause
disp('  ')
disp('The figure presents magnitude responses produced from time-domain responses')
disp('R=mixkautz1(a,2000,1). The filter order is now 40, i.e., each resonant') 
disp('frequency is represented by two tap-output signals. (For your own pole')
disp('vector choice, the filter order may be less than double the pole vector')
disp('dimension, depending on the pole distribution, and truncation effects may')
disp('occure. More presicely, the filter order is')
disp('length(a(find(imag(a)==0)))+2*length(a(find(imag(a)>0)))).')
%
if i==5
   a=px;
   C=mixkautz1(a,2000,1);
else
   a=p(:,i);
   C=mixkautz1(a,2000,1);
end
%
M=abs(fft(C));
subplot(2,1,1)
f1=linspace(0,1,1000);
plot(f1,20*log10(M(1:1000,:)))
title('Magnitude responses on a linear frequency scale')
xlabel('Normalized frequency')
ylabel('Magnitude / dB')
axis([0 1 -30 30])
subplot(2,1,2)
semilogx(f1,20*log10(M(1:1000,:)))
title('Magnitude responses on a logarithmic frequency scale')
xlabel('Normalized frequency')
ylabel('Magnitude / dB')
axis([.01 1 -30 30])
%
disp('  ')
disp('The nice symmetry of the complex structure is gone. This is due to the')
disp('additional zeros at -1 and 1 in the 2nd order blocks of this particular')
disp('real Kautz filter structure. Other choices are also possible.')
disp(' ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('(At least for the given pole sets) this can be further demonstrated by')
disp('separating the tap-outputs with odd and even indexes (i.e. columns in R):')
disp('there are two sets of functions assiciated to the given frequencies, with')
disp('different magnitude response shapes, especially at the band edges.')
%
[m,n]=size(C);
subplot(2,1,1)
plot(f1,20*log10(M(1:1000,1:2:n)))
title('Magnitude responses of odd tap-outputs')
xlabel('Normalized frequency')
ylabel('Magnitude / dB')
axis([0 1 -30 30])
subplot(2,1,2)
plot(f1,20*log10(M(1:1000,2:2:n)))
title('Magnitude responses of even tap-outputs')
xlabel('Normalized frequency')
ylabel('Magnitude / dB')
axis([.01 1 -30 30])
%
disp(' ')
disp('.... press any key to continue ...')
pause
disp('   ')
disp('The response of the complex or real Kautz filter is simply y=S*w,')
disp('where w is a coefficient vector (with appropriate dimensions) and')
disp('S is the signal matrix produced by input x (e.g. S=mixkautz1(a,N,x)).')
disp('  ')
disp('There are many interpretations and methods for input-output-data')
disp('identification of the filter coefficients w, also in the Kautz model') 
disp('case. The prototype least-square (LS) solution is attained from the')
disp('normal equations: S''*S*w=S''*d, where d is the desired response.')
disp(' ')
disp('Here we address only the approximation problem, i.e., weighing of the')
disp('tap-output impulse responses: h=S*c. Here too we have many interpretations')
disp('and optimization criteria for c. We use LS or Fourier coefficients because')
disp('they are the true orthonormal expansion coefficients, and they are easy to')
disp('get: c=S(N,:), with S=mixkautz1(a,N,x(N:-1:1)). This implements time-domain')
disp('convolutions and it is "analogous" to rectangular window FIR design.')
disp(' ')
disp('.... press any key to continue ...')
pause
disp('  ')
%
A=abs(fft(ak2));
[c,e,h]=kautzapprx1(a,ak2);
H=abs(fft(h));
subplot(2,1,1)
plot(t,[ak2 h])
title('The target response and an approximation')
xlabel('Time / sec')
ylabel('Amplitude')
subplot(2,1,2)
f=linspace(0,12000,2400);
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Corresponding magnitude responses')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 6000 -50 25])
%
disp('In the figure a measured acoustic guitar body impulse response is')
disp('modeled with your choice of Kautz filter structure. It''s probably')
disp('not looking too promising because of the particular choice of poles')
disp('and a relatively low filter order.')
disp(' ')
disp('The approximation was composed using [c,E,h]=kautzappx1(a,x), where')
disp('h is the approximation of x produced by mixkautz1.m. c are the Kautz')
disp('filter coefficients and E is the normalized RMS error of x-h.')
disp('  ')
disp('.... press any key to continue ...')
pause
disp('  ')
disp('Actually some of the given pole distributions wouldn''t be that bad')
disp('if we increase the filter order and drop some poles at the band edges.')
disp('We get e.g. a 61th order model from the logarithmically spaced pole')
disp('set with inversely proportional radii, presented in the figure.')
%
[c,e,h]=kautzapprx1(logaki100(15:80),ak2);
H=abs(fft(h));
subplot(2,1,1)
plot(t,[ak2 h])
title('The target response and an 61th order Kautz approximation')
xlabel('Time / sec')
ylabel('Amplitude')
subplot(2,1,2)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Corresponding magnitude responses')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 6000 -50 25])
disp(' ')
disp('.... press any key to continue ...')
pause
disp('  ')
disp('There are basically three strategies in choosing a Kautz filter structure:')
disp('  ')
disp('   * to use simple choices, e.g. one pole pair or a fixed pole distribution')
disp('   * fitting poles to the resonances (in time and/or frequency domains)')
disp('   * choosing a substructure and using it repetitively')
disp(' ')
pause(2)
[c,E,h]=kautzapprx2([.98 .95 .9 .95 .9]'.*exp(j*pi*res5freq),20,ak2);
H=abs(fft(h));
subplot(2,1,1)
plot(t,[ak2 h])
title('The target response and an 200th order Kautz approximation')
xlabel('Time / sec')
ylabel('Amplitude')
subplot(2,1,2)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Corresponding magnitude responses and selected resonances')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 5000 -50 25])
p=res5freq*12000;						% adding lines to the figure
L=[-35;-15]*ones(1,5);					% to indicate pole pair positions
line([p';p'],L);
text(200,-40,'chosen pole pairs')
%
disp('The actual methods may include manual tuning of resonances, all-pole')
disp('or zero-pole modeling, sophisticated guesses, and random or iterative')
disp('search. To demonstrate the last approach we have located five prominent')
disp('resonances and estimated the corresponding pole radii. Using 20 cascaded')
disp('blocks we get the 200th order Kautz approximation in the figure.')
%
disp(' ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('The approximation was composed using [c,E,h]=kautzapprx2(a,M,x), which')
disp('produces the same h as kautzapprox1.m with a=a*ones(1,M), a=a(:), but')
disp('it preserves the structure of identical block, i.e. the order in c.')
disp('According to orthogonality, we then get the approximation error associated')
disp('to the number of blocks used in the approximation, i=1:M, in the form:')
disp('e(i)=x''*x-c(1:p*i)''*c(1:p*i), where p is the suborder. This makes')
disp('model reduction trivial by simply inspecting the error curve (figure).')
%
clf
plot(E)
xlabel('Block orders, i.e. filter orders in decades')
ylabel('Normalized RMS approximation errors')
title('Kautz approximation errors for orders 10 x i ,  i=1:20')
disp(' ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('For example block order 8 would give the following approximation')
disp('at a considerably reduced filter order 80.')
%
[c,E,h]=kautzapprx2([.99 .98 .95 .97 .95]'.*exp(j*pi*res5freq),8,ak2);
H=abs(fft(h));
subplot(2,1,1)
plot(t,[ak2 h])
title('The target response and an 80th order Kautz approximation')
xlabel('Time / sec')
ylabel('Amplitude')
subplot(2,1,2)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Corresponding magnitude responses and selected resonances')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 5000 -50 25])
p=res5freq*12000;						% adding lines to the figure
L=[-35;-15]*ones(1,5);					% to indicate pole pair positions
line([p';p'],L);
text(200,-40,'chosen pole pairs')
disp(' ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('It''s a bit surprising that standard all-pole or pole-zero modeling do not')
disp('in general provide good pole sets, apart from the Steiglitz-McBride (SM)')
disp('method. Actually this observation clarifies the situation: we don''t want an')
disp('overall spectral model or a coupled pole-zero structure. We have adopted')
disp('a method, proposed originally to pure  FIR-to-IIR fitler conversion, to the') 
disp('pole position optimization of a real Kautz structure. It resembles the SM')
disp('method, but it genuinely optimizes the pole positions of the orthonormal')
disp('expansion, and not just the poles of a coupled rational overall structure.')
disp('  ')
disp('Function [E,Q]=braun1(N,I,x) uses I iterations to search for an Nth order')
disp('Kautz structure to approximate x. E(i) is the NRMS error and Q(i,:) are') 
disp('the (Kautz filter common denominator) polynomial coefficients, for i=1:I.')
disp('  ')
disp('For higher orders, braun1.m will cause warnings and take time. E may peak')
disp('to NaN and Inf but we should always get stable and somewhat (theoretically')
disp('globally) optimal pole sets for sufficient I.')
disp(' ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('We use roots(Q(k,:)), corresponding to minimum E from [m,k]=min(E),') 
disp('as the Kautz filter poles. Results for orders N=4, 8, 10 and 16 and') 
disp('I=30 are on their way to the figure.')
pause(1)
%
%disp(' ')
%disp('Warning: Subsequent singularity warning messages are disregarded')
warning off
clf
ind=[4 8 10 16];
for i=1:4
   pause(.5)
   [E Q]=braun1(ind(i),30,ak2);
   [m,k]=min(E);
   subplot(2,2,i)
   p=roots(Q(k,:));
   zplane(p);
   if i==1
      title('orders 4 and 10')
   elseif i==2
      title('orders 8 and 16')
      p8=p;
   elseif i==3   
      p10=p;
   end   
end
warning on
disp('  ')
disp('.... press any key to continue ...')
pause
disp('  ')
disp('These pole sets can be used directly to define substructures but an')
disp('ad hoc damping method smoothes the response in off-resonance regions:')
disp('we simply raise the pole radii to the power of the block number.')
disp('  ')
disp('Results for suborders 8 and 10, and filter orders 96 and 100, respectively,')
disp('are displayed in the figure along with lines indicating the pole positions.')
%
p8=(abs(p8).^8).*exp(j*angle(p8));
[c,E,h]=kautzapprx2(p8,16,ak2);
H=abs(fft(h));
subplot(2,1,1)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Magnitude response of an (8 x 16) order Kautz approximation')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 5000 -50 25])
p=angle(p8)*12000/pi;						% adding lines to the figure
L=[-35;-15]*ones(1,8);					% to indicate pole pair positions
line([p';p'],L);
%
p10=(abs(p10).^10).*exp(j*angle(p10));
[c,E,h]=kautzapprx2(p10,10,ak2);
H=abs(fft(h));
subplot(2,1,2)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Magnitude response of an (10 x 10) order Kautz approximation')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 5000 -50 25])
p=angle(p10)*12000/pi;						% adding lines to the figure
L=[-35;-15]*ones(1,10);					% to indicate pole pair positions
line([p';p'],L);
%
disp(' ')
disp('.... press any key to continue ...')
pause
disp('  ')
disp('To demonstrate that braun1.m is able to capture the whole resonance')
disp('structure, the figure presents an 102th order Kautz approximation')
disp('composed from a pre-calculated 120th order pole set, omitting some')
disp('of the poles (trying to model the anti-aliasing filter). Notice that')
disp('there are only two weights per pole pair and that some of the weaker')
disp('poles have very little contribution to the spectral model.')
disp(' ')
%
p=bu120ak2(19:120);
[c,E,h]=kautzapprx1(p,ak2);
H=abs(fft(h));
subplot(2,1,1)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Magnitude response of an 102th order Kautz approximation')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 6000 -50 25])
p=angle(p)*12000/pi;						% adding lines to the figure
L=[-35;-15]*ones(1,102);					% to indicate pole pair positions
line([p';p'],L);
%
subplot(2,1,2)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('The low frequency region of the same response ')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([10 2000 -50 25])
line([p';p'],L);
%
disp('.... press any key to continue ...')
pause
disp(' ')
disp('The fact that a high order distributed pole set is a good representation')
disp('of the overall resonance structure can then again be used as a basis for')
disp('substructures. In fact, poles obtained by pruning a large set are more')
disp('accurately "radius tuned" than those obtained directly on a suborder.')
disp('  ')
disp('We conclude with two selections of relevant resonances from the 120th')
disp('order pole set. Pole angles associated with 5 prominent resonances are')
disp('almost exactly equal to those obtained previously by a peak search method.')
disp('In the second pole set we have added 7 pole pairs corresponding to some')
disp('resonances in the frequency region from 1400 to 3600 Hz.')
%
b=bu120ak2([100 110 112 116 114]);	% five prominent resonances
b=(abs(b).^10).*exp(j*angle(b));		% pole radius damping
[c,E,h]=kautzapprx2(b,10,ak2);
H=abs(fft(h));
subplot(2,1,1)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Magnitude response of an (10 x 10) order Kautz approximation and pole positions')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([0 4500 -50 25])
p=angle(b)*12000/pi;						% adding lines to the figure
L=[-35;-15]*ones(1,5);					% to indicate pole pair positions
line([p';p'],L);
%
b=bu120ak2([78 82 84 88 90 96 98 100 110 112 116 114]);	% 12 prominent resonances
b=(abs(b).^4).*exp(j*angle(b));		% pole radius damping
[c,E,h]=kautzapprx2(b,4,ak2);
H=abs(fft(h));
subplot(2,1,2)
plot(f,20*log10([A(1:2400) H(1:2400)]))
title('Magnitude response of an (24 x 4) order Kautz approximation and pole positions')
xlabel('Frequency / Hz')
ylabel('Magnitude / dB')
axis([10 4500 -50 25])
p=angle(b)*12000/pi;						% adding lines to the figure
L=[-35;-15]*ones(1,12);					% to indicate pole pair positions
line([p';p'],L);
%
disp('  ')
disp('.... press any key to continue ...')
pause
disp(' ')
disp('That''s all! Thank you for your interest. We would like to emphasize that')
disp('this was just an bumpy introduction to Kautz filter design and not') 
disp('a case study on acoustic guitar body response modeling.')
disp(' ')
disp('.... press any key to exit ...')
pause
disp('  ')

% Revised November 29, 2000, Tuomas Paatero
% Revised January 5, 2001, Tuomas Paatero
