function S=mixkautz1(a,N,X)

% S=mixkautz1(a,N,X): Tap-output signals of a real Kautz filter, 
% defined by pole vector a, excited by X, using a particular mixed 
% structure for real and complex conjugate pole pairs. 
%
% N is the desired length of the output, if N > length(X), 
% and tap-output signals are the columns of S. 
%
% Note that a is replaced by a=a(find(imag(a)>=0)) to avoid dublication 
% produced by complex conjugate poles, so check the resulting filter order. 
%
% Examples:	
%
%* G=mixkautz1(a,N,1) are the (length N) impulse responses
%		of Kautz functions defined by a.
%
%* Orthonormatity could be tested by S'*S=I (for sufficient N).
%
%* y=S*w, S=mixkautz(a,N,x), is the Kautz filter response 
%		to input x with coefficient vector w.
%
%* c=S(N,:), N=length(X), are the Fourier coefficients produced
%	by feeding invX=X(N:-1:1) to S=mixkautz(a,N,invX) and reading
%	tap-outputs at N. These are the LS optimal coefficients
%	for approximating x with y=G*c.

if length(X)<N,
	X=[X;zeros(N-length(X),1)]; 	% zero-padding of X to length N
end

a=a(:);						% Poles with imag(a)<0 are disregarded to avoid
a=a(find(imag(a)>=0));	% dublication for conjugate pole pairs in a.

ar=a(find(imag(a)==0)); 	% real poles
ak=a(find(imag(a)~=0)); 	% complex poles
m=length(ar);					
n=length(ak);
q=-2*real(ak);		% calculation of the 2nd order
r=abs(ak).^2;		% polynomial terms
A=X;					% the first AP filter is 1

for i=1:m														% first we calculate
   S(:,i)=sqrt(1-ar(i)^2)*filter(1,[1 -ar(i)],A); 	% block's related to
   A=filter([-ar(i) 1],[1 -ar(i)],A);					% real poles
end
% and then tap-output pairs corresponding to complex (conjugate) poles
for i=1:n
	S(:,m+2*i-1)=sqrt((1-r(i))*(1+r(i)-q(i))/2)*filter([1 -1],[1 q(i) r(i)],A);
	S(:,m+2*i)=sqrt((1-r(i))*(1+r(i)+q(i))/2)*filter([1 1],[1 q(i) r(i)],A);   
   A=filter([r(i) q(i) 1],[1 q(i) r(i)],A);
end


% Revised November 7, 2000, Tuomas Paatero
