/* vbap.c  version 0.1
(c) Ville Pulkki   2.2.1999 Helsinki University of Technology*/


#include <stdio.h>
#include <dmedia/audio.h>
#include <dmedia/audiofile.h>
#include <curses.h>
#include <math.h>

#define BUF_SIZE 40000
#define MAX_TRIPLET_AMOUNT 60
#define GAIN 1.0
#define INCR 2


/* This is a program which pans a monophonic aiff file to eight channels
using Vector Base Amplitude Panning (VBAP), 
see Pulkki: "Virtual Source Positioning using VBAP" 
in  Journal of the Audio Engineering Society June 1997.
The program can be easily modified to handle multiple audio input
channels separately. This program is just to demonstrate how 
VBAP can be implemented. 

Calculate the loudspeaker triplet (or pair) matrices first using 
program define_loudspeakers. 

The position of the virtual source can be controlled using cursor keys. 

Since the virtual source may be moving, the gain factors are 
calculated after each writing of 8-channel audio buffer. 
Two sets of gain factors are interpolated between previous 
and current gain factors. One set is always faded out and one set is 
faded in. This yields completely click-free panning, though requires now
6 multiplications per a sound sample. This can be optimized in many cases to 
3 multiplications per a sound sample. If you plan to do it, remember to 
handle situations in which the triplet is changed correctly. 

Current implementation is capable handling both 3-D and 2-D 
loudspeaker configurations. The configured loudspeaker matrices define 
the dimensions.
*/


/* globals */

double lsm[MAX_TRIPLET_AMOUNT][9]; /* loudspeaker triplet matrices */
int lstripl[MAX_TRIPLET_AMOUNT][3]; /* loudspeaker triplet ldspeaker numbers */
int triplet_amount;
int dimension;

double *angle_to_cart(int azi, int ele)
{
  double *res;
  double atorad = (2 * 3.1415927 / 360) ;
  res = (double *) malloc(3*sizeof(double));
  res[0] = (float) (cos((double) (azi * atorad)) * cos((double)  (ele * atorad)));
  res[1] = (float) (sin((double) (azi * atorad)) * cos((double) (ele * atorad)));
  res[2] = (float) (sin((double) (ele * atorad)));
  return res;
}


void vbap(double g[3], int ls[3], int azi, int ele) 
{
  /* calculates gain factors using loudspeaker setup and given direction */
  double *cartdir;
  double power;
  int i,j,k;
  double small_g;
  double big_sm_g, gtmp[3];
  int winner_triplet;

  cartdir=angle_to_cart(azi,ele);  
  big_sm_g = -100000.0;
  for(i=0;i<triplet_amount;i++){
    small_g = 10000000.0;
    for(j=0;j<dimension;j++){
      gtmp[j]=0.0;
      for(k=0;k<dimension;k++)
	gtmp[j]+=cartdir[k]*lsm[i][k+j*dimension]; 
      if(gtmp[j] < small_g)
	small_g = gtmp[j];
    }
    if(small_g > big_sm_g){
      big_sm_g = small_g;
      winner_triplet=i;
      g[0]=gtmp[0]; g[1]=gtmp[1]; 
      ls[0]=lstripl[i][0]; ls[1]=lstripl[i][1]; 
      if(dimension==3){
	g[2]=gtmp[2];
	ls[2]=lstripl[i][2];
      } else {
	g[2]=0.0;
	ls[2]=0;
      }
    }
  }
  
  power=sqrt(g[0]*g[0] + g[1]*g[1] + g[2]*g[2]);
  /*  power=g[0]+g[1];*/
  
  g[0] /= power; 
  g[1] /= power;
  g[2] /= power;
  free(cartdir);
}

void read_ls_conf(FILE *fp ){
  /* reads from specified file the loudspeaker triplet setup */
  int amount,i,j,a,b,d=0;
  char *toke;
  char c[1000];
  double mx[9];
  fgets(c,1000,fp);
  toke = (char *) strtok(c, " ");
  toke = (char *) strtok(NULL, " ");
  toke = (char *) strtok(NULL, " ");
  if((toke = (char *) strtok(NULL, " "))==NULL){
    fprintf(stderr,"Wrong ls matrix file?\n");
    exit(-1);
  }
  sscanf(toke, "%d",&amount);
  toke = (char *) strtok(NULL, " ");
  toke = (char *) strtok(NULL, " ");
  if((toke = (char *) strtok(NULL, " "))==NULL){
    fprintf(stderr,"Wrong ls matrix file?\n");
    exit(-1);
  }
  sscanf(toke, "%d",&dimension);
  printf("dim %d\n",dimension);
  triplet_amount = amount;
  for(i=0;i<amount;i++){
    fgets(c,1000,fp);
    toke = (char *) strtok(c, " "); 
    if(strncmp(toke,"Trip",4)!=0 && dimension==3){
      fprintf(stderr,"Something wrong in ls matrix file\n");
      exit(-1);
    }
    if(strncmp(toke,"Pair",4)!=0 && dimension==2){
      fprintf(stderr,"Something wrong in ls matrix file\n");
      exit(-1);
    }
    toke = (char *) strtok(NULL, " "); 
    toke = (char *) strtok(NULL, " "); toke = (char *) strtok(NULL, " ");
    sscanf(toke, "%d",&a);
    lstripl[i][0]=a; 
    toke = (char *) strtok(NULL, " ");
    sscanf(toke, "%d",&b);
    lstripl[i][1]=b; 
    if (dimension==3){
      toke = (char *) strtok(NULL, " ");
      sscanf(toke, "%d",&d);
      lstripl[i][2]=d;
    }
    if(a>8 || b >8 || d>8){
      fprintf(stderr,"Can't currently handle more than 8 channels!\n");
      exit(-1);
    }

    toke = (char *) strtok(NULL, " ");
    for(j=0;j<(dimension*dimension);j++){
      toke = (char *) strtok(NULL, " ");
      sscanf(toke, "%lf",&(mx[j]));
      lsm[i][j]=mx[j];
    }
  }
}

main(int argc, char **argv) 
{
 ALport p;
 short buf[BUF_SIZE*16];
 short *inbuf;
 long a,i,j;
 long frames;
 ALconfig c;
 AFfilehandle fh;
 int button;
 int azi=0,ele=0,h;
 int counter=0;
 short obuf[9],x;
 short *optr0,*optr1,*optr2,*optr3,*optr4,*optr5;
 int triple_ls_1, triple_ls_2, triple_ls_3, target_ls;
 FILE *fp;
 float ascscale,descscale;
 float g0,g1,g2,g3,g4,g5;
 float go0,go1,go2,go3,go4,go5;
 double g[3];
 int ls[3];
 WINDOW *w;
 char tmp[100];


 if(argc != 3){
   fprintf(stderr,"Usage: vbap aiff_file ls_set_matrix_file\n");
   exit(-1);
 }

 /* opening the sound file */
 fh=afOpenFile(argv[1],"r",0);
 frames=AFgetframecnt(fh,AF_DEFAULT_TRACK);
 if(afGetChannels(fh, AF_DEFAULT_TRACK) != 1){
   fprintf(stderr, "Supports only mono aiff-files\n");
   exit(-1);
 }
 inbuf= malloc(frames*sizeof(short)*2);
 afReadFrames(fh,AF_DEFAULT_TRACK,inbuf,frames);

 /*opening the audio port*/
 c = alNewConfig();
 if (!c) {
   printf("Couldn't create ALconfig:%s\n", alGetErrorString(oserror()));
   exit(-1);
 }
 alSetChannels(c,8);
 ALsetqueuesize(c,BUF_SIZE);
 p = alOpenPort("alWriteFrames example","w",c);
 if (!p) {
   printf("port open failed:%s\n", alGetErrorString(oserror()));
   exit(-1);
 }

 /* opening the loudspeaker matrix file*/
 if((fp=fopen(argv[2],"r"))==NULL){
   fprintf(stderr,"Could not loudspeaker data file %s\n",argv[6]);
   exit(0);
 }
 read_ls_conf(fp);

 /* screen initialization */
 w=initscr();
 cbreak();
 noecho();
 keypad(w,TRUE);
 nodelay(w,TRUE);

 sprintf(tmp,"%d-D VBAP demonstration v 0.1\n",dimension);    mvaddstr(3,3,tmp);
 sprintf(tmp,"Sound file: %s\n",argv[1]);  mvaddstr(6,6,tmp);
 sprintf(tmp,"Virtual source direction:\n");  mvaddstr(8,6,tmp);
 sprintf(tmp,"(moves with cursor keys)\n");  mvaddstr(14,8,tmp);
 sprintf(tmp,"Azimuth   \n");          mvaddstr(10,8,tmp);
 sprintf(tmp,"Elevation \n");          mvaddstr(12,8,tmp);
 sprintf(tmp,"Loudspeakers\n");          mvaddstr(16,6,tmp);
 sprintf(tmp,"Gains\n");          mvaddstr(17,6,tmp);

 /* vbap initialization */

 while(1){ /* finding a proper starting direction */
   vbap(g, ls, azi, ele);
   if(g[0]<-0.01 || g[1]<-0.01 || g[2]<-0.01 ){
     azi=azi+2;
     if(azi>180){
       azi=-178; ele=ele+5;
       if(ele>90)
	 ele=-90;
     }
     if(ele==-5 && azi == -2)
       break;
   } else
     break;
 }

 /*pointers for output*/
 optr0=&(obuf[ls[0]]);
 optr1=&(obuf[ls[1]]);
 optr2=&(obuf[ls[2]]);
 g0=go0=(float) g[0]; g1=go1=(float) g[1]; g2=go2=(float) g[2];

 sprintf(tmp,"Azimuth   %d\n",azi);
 mvaddstr(10,8,tmp);
 sprintf(tmp,"Elevation %d\n",ele);
 mvaddstr(12,8,tmp);
 sprintf(tmp," %d    %d    %d\n",ls[0],ls[1],ls[2]);
 mvaddstr(16,20,tmp);
 sprintf(tmp,"%1.2f %1.2f %1.2f\n",g[0],g[1],g[2]);
 mvaddstr(17,20,tmp);



 /* main loop */
 while(1){
   a=0;
   for(j=0;j<(1+frames/BUF_SIZE);j++){
     while((button=getch())!=ERR){
       if(button==KEY_LEFT) {azi = azi - INCR;}
       if(button==KEY_RIGHT){azi = azi + INCR;}
       if(dimension==3){
	 if(button==KEY_UP)   {ele = ele + INCR;}
	 if(button==KEY_DOWN) {ele = ele - INCR;}
       }
       if(button==KEY_ENTER){
	 alClosePort(p);
	 endwin(); exit(1);
       }
       vbap(g, ls, azi, ele);  
       g[0]*=GAIN; g[1]*=GAIN;  g[2]*=GAIN; 

       /* If virtual source outside defined area, take movement back */
       if(g[0]<-0.01 || g[1]<-0.01 || g[2]<-0.01 ){
	 if(button==KEY_LEFT)  azi = azi + INCR;
	 if(button==KEY_RIGHT) azi = azi - INCR ;
	 if(dimension==3){
	   if(button==KEY_UP)    ele = ele - INCR; 
	   if(button==KEY_DOWN)  ele = ele + INCR; 
	 }
	 vbap(g, ls, azi, ele);   
	 g[0]*=GAIN; g[1]*=GAIN;  g[2]*=GAIN; 
       }

       /* keep -180 < azi < 180 and -90 < ele < 90 */
       if(azi>180) azi-=360;
       if(azi<-180) azi+=360;
       if(ele>90) ele=90;
       if(ele<-90) ele=-90;

       /* update screen */
       sprintf(tmp,"Azimuth   %d\n",azi);
       mvaddstr(10,8,tmp);
       sprintf(tmp,"Elevation %d\n",ele);
       mvaddstr(12,8,tmp);
       sprintf(tmp," %d    %d    %d\n",ls[0],ls[1],ls[2]);
       mvaddstr(16,20,tmp);
       sprintf(tmp,"%1.2f %1.2f %1.2f\n",g[0],g[1],g[2]);
       mvaddstr(17,20,tmp);
     }
     
 
     /* sound output */
     optr3=&(obuf[ls[0]]);
     optr4=&(obuf[ls[1]]);
     optr5=&(obuf[ls[2]]);
     go3=(float) g[0]; go4=(float) g[1]; go5=(float) g[2];
     g3=0.0;g4=0.0;g5=0.0;
     i=0;
     while(i< (BUF_SIZE*8)){
       for(h=1;h<9;h++)
	 obuf[h]=0;
       x= (float) inbuf[a];

       /* panning */
       *optr0=(short)(x*g0);*optr1=(short)(x*g1);*optr2=(short)(x*g2);
       (*optr3) +=(short)(x*g3);(*optr4)+=(short)(x*g4);(*optr5)+=(short)(x*g5);

       /* writing samples to buffers */
       buf[i++]=obuf[1];buf[i++]=obuf[2]; buf[i++]=obuf[3];buf[i++]=obuf[4];
       buf[i++]=obuf[5];buf[i++]=obuf[6]; buf[i++]=obuf[7];buf[i++]=obuf[8]; 
       a++;
       if (a>=frames)
	 a=frames-1; 

       /* interpolation */
       if((a%5)==0){ /* interpolate gains after each 5th sample */
	 ascscale=((float) i / (float) (BUF_SIZE*8));
	 descscale= 1.0 - ascscale;
	 g0=go0*descscale; g1=go1*descscale;g2=go2*descscale;
	 g3=go3*ascscale; g4=go4*ascscale;g5=go5*ascscale;
       }
     }
     go0=g0=g3; go1=g1=g4; go2=g2=g5;
     optr0=optr3;optr1=optr4;optr2=optr5;
     alWriteFrames(p, buf, BUF_SIZE); /* write BUF_SIZE 8-channel frames */
   }
   free(inbuf);
 }
}
